<?php
// etseo_engine_schema.php
declare(strict_types=1);

/**
 * Schema STRICT pentru răspunsul AI (Structured Outputs).
 */
function etseo_engine_output_schema(): array {
    return [
        "name" => "etseo_safe_output",
        "schema" => [
            "type" => "object",
            "additionalProperties" => false,
            "properties" => [
                "ok" => ["type" => "boolean"],
                "risk_level" => ["type" => "string", "enum" => ["zero", "mic", "mediu", "mare"]],
                "page_type" => ["type" => "string", "enum" => ["Article", "Product", "Service", "WebPage"]],
                "meta_title" => ["type" => "string"],
                "meta_description" => ["type" => "string"],
                "og_title" => ["type" => "string"],
                "og_description" => ["type" => "string"],
                "tw_title" => ["type" => "string"],
                "tw_description" => ["type" => "string"],
                "canonical" => ["type" => "string"],
                "json_ld" => ["type" => "string"],
                "alt_text_map" => [
                    "type" => "object",
                    "additionalProperties" => ["type" => "string"]
                ]
            ],
            "required" => [
                "ok","risk_level","page_type",
                "meta_title","meta_description",
                "og_title","og_description",
                "tw_title","tw_description",
                "canonical","json_ld"
            ]
        ],
        "strict" => true
    ];
}

/**
 * Generator SAFE de JSON-LD (folosit ca fallback/enforcement).
 */
function etseo_engine_schema_for_page(string $pageType, string $url, string $title, string $desc, array $ctx = []): string {
    $url  = trim($url);
    $title = trim($title);
    $desc  = trim($desc);

    $base = [
        "@context" => "https://schema.org",
        "url" => $url,
    ];

    if ($pageType === 'Product') {
        $data = $base + [
            "@type" => "Product",
            "name" => $title,
            "description" => $desc,
        ];
        if (!empty($ctx['brand'])) {
            $data["brand"] = ["@type" => "Brand", "name" => (string)$ctx['brand']];
        }
        return json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    }

    if ($pageType === 'Service') {
        $data = $base + [
            "@type" => "Service",
            "name" => $title,
            "description" => $desc,
        ];
        if (!empty($ctx['provider_name'])) {
            $data["provider"] = ["@type" => "Organization", "name" => (string)$ctx['provider_name']];
        }
        return json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    }

    $data = $base + [
        "@type" => "Article",
        "mainEntityOfPage" => [
            "@type" => "WebPage",
            "@id" => $url
        ],
        "headline" => $title,
        "description" => $desc,
    ];

    return json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
}
