<?php
// etseo_engine_safe_generate.php
declare(strict_types=1);

require_once __DIR__ . '/etseo_engine_config.php';
require_once __DIR__ . '/etseo_engine_helpers.php';
require_once __DIR__ . '/etseo_engine_ai_client.php';
require_once __DIR__ . '/etseo_engine_schema.php';
require_once __DIR__ . '/etseo_engine_cache.php';


etseo_engine_cors_headers();

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') {
    etseo_engine_respond(['ok' => true]);
}

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
    etseo_engine_respond(['ok' => false, 'error' => 'Method not allowed'], 405);
}

etseo_engine_require_auth_if_set();
etseo_engine_rate_limit('safe-generate', 60);


$in = etseo_engine_read_json();

$url       = isset($in['url']) ? (string)$in['url'] : '';
$title     = isset($in['title']) ? (string)$in['title'] : '';
$excerpt   = isset($in['excerpt']) ? (string)$in['excerpt'] : '';
$content   = isset($in['content']) ? (string)$in['content'] : '';
$post_type = isset($in['post_type']) ? (string)$in['post_type'] : 'page';

$url = etseo_engine_make_canonical($url);
if ($url === '') {
    etseo_engine_respond(['ok' => false, 'error' => 'Missing url'], 400);
}

// ------------- Fallback (heuristic) -------------
// SAFE MODE heuristic (fallback) – funcționează și dacă AI e oprit/pică
$baseTitle  = $title !== '' ? $title : 'Pagina';
$meta_title = etseo_engine_clean_text($baseTitle, 58);

// descriere: excerpt dacă există, altfel din content
$descSource = $excerpt !== '' ? $excerpt : $content;
$meta_desc  = etseo_engine_clean_text($descSource, 155);
if ($meta_desc === '') {
    $meta_desc = 'Descoperă informațiile esențiale din această pagină și îmbunătățește vizibilitatea în Google.';
}

$canonical = $url;

$cacheKey = etseo_engine_cache_key($canonical, $post_type);
$cached = etseo_engine_cache_get($cacheKey, 86400);
if (is_array($cached) && ($cached['ok'] ?? false)) {
    etseo_engine_respond($cached);
}


$og_title = $meta_title;
$og_desc  = $meta_desc;
$tw_title = $meta_title;
$tw_desc  = $meta_desc;

require_once __DIR__ . '/etseo_engine_schema.php';

$page_type_fallback = ($post_type === 'product') ? 'Product' : 'Article';
$json_ld = etseo_engine_schema_for_page($page_type_fallback, $canonical, $meta_title, $meta_desc);


$fallback = [
    'ok' => true,
    'risk_level' => 'zero',
    'page_type' => ($post_type === 'product' ? 'Product' : 'Article'),

    'meta_title' => $meta_title,
    'meta_description' => $meta_desc,

    'og_title' => $og_title,
    'og_description' => $og_desc,

    'tw_title' => $tw_title,
    'tw_description' => $tw_desc,

    'canonical' => $canonical,
    'json_ld' => $json_ld,
];

// ------------- AI SAFE MODE -------------
// Încearcă să genereze prin AI (Structured JSON). Dacă pică, revenim la fallback.
$ai = etseo_engine_openai_safe_generate($in);

if (!is_array($ai) || !($ai['ok'] ?? false)) {
    etseo_engine_cache_set($cacheKey, $fallback);
    etseo_engine_respond($fallback);
}


// Enforce SAFE RULES (nu lăsăm AI să schimbe canonical / să ceară risc)
$ai['canonical'] = $canonical;

// acceptăm doar risk_level: zero/mic
$rl = (string)($ai['risk_level'] ?? 'mic');
if ($rl !== 'zero' && $rl !== 'mic') {
    $ai['risk_level'] = 'mic';
}

// dacă lipsesc câmpuri (ar trebui să nu se întâmple cu schema strictă), completăm safe
$ai['meta_title'] = isset($ai['meta_title']) ? (string)$ai['meta_title'] : $fallback['meta_title'];
$ai['meta_description'] = isset($ai['meta_description']) ? (string)$ai['meta_description'] : $fallback['meta_description'];

$ai['og_title'] = isset($ai['og_title']) ? (string)$ai['og_title'] : $ai['meta_title'];
$ai['og_description'] = isset($ai['og_description']) ? (string)$ai['og_description'] : $ai['meta_description'];

$ai['tw_title'] = isset($ai['tw_title']) ? (string)$ai['tw_title'] : $ai['meta_title'];
$ai['tw_description'] = isset($ai['tw_description']) ? (string)$ai['tw_description'] : $ai['meta_description'];

$ai['page_type'] = isset($ai['page_type']) ? (string)$ai['page_type'] : $fallback['page_type'];

// json_ld trebuie să fie string JSON valid
if (!isset($ai['json_ld']) || !is_string($ai['json_ld']) || trim($ai['json_ld']) === '') {
    $ai['json_ld'] = $fallback['json_ld'];
} else {
    $tmp = json_decode($ai['json_ld'], true);
    if (!is_array($tmp)) {
        $ai['json_ld'] = $fallback['json_ld'];
    }
}

$ai['ok'] = true;

etseo_engine_cache_set($cacheKey, $ai);


etseo_engine_respond($ai);
