<?php
// etseo_engine_helpers.php
declare(strict_types=1);

function etseo_engine_read_json(): array {
    $raw = file_get_contents('php://input');
    if ($raw === false || trim($raw) === '') return [];
    $json = json_decode($raw, true);
    return is_array($json) ? $json : [];
}

function etseo_engine_require_auth_if_set(): void {
    if (ETSEO_ENGINE_API_KEY === '') return;

    $auth = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
    $auth = is_string($auth) ? trim($auth) : '';
    $expected = 'Bearer ' . ETSEO_ENGINE_API_KEY;

    if ($auth !== $expected) {
        etseo_engine_respond(['ok' => false, 'error' => 'Unauthorized'], 401);
    }
}

function etseo_engine_clean_text(string $s, int $maxLen): string {
    $s = trim(preg_replace('~\s+~u', ' ', strip_tags($s)));
    if (mb_strlen($s) > $maxLen) $s = mb_substr($s, 0, $maxLen - 1) . '…';
    return $s;
}

function etseo_engine_make_canonical(string $url): string {
    $url = trim($url);
    if ($url === '') return '';
    // elimină #fragment
    $url = preg_replace('~#.*$~', '', $url);
    return $url;
}

function etseo_engine_jsonld_article(string $url, string $headline, string $desc): string {
    $data = [
        "@context" => "https://schema.org",
        "@type" => "Article",
        "mainEntityOfPage" => [
            "@type" => "WebPage",
            "@id" => $url
        ],
        "headline" => $headline,
        "description" => $desc,
        "url" => $url
    ];
    return json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
}


function etseo_engine_rate_limit(string $bucket, int $maxPerMinute = 30): void {
    $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    $key = hash('sha256', $bucket . '|' . $ip . '|' . date('Y-m-d H:i'));
    $dir = rtrim(getenv('HOME') ?: sys_get_temp_dir(), '/') . '/tmp/etseo_engine_rl';
    if (!is_dir($dir)) @mkdir($dir, 0775, true);
    $file = $dir . '/' . $key . '.txt';

    $count = 0;
    if (is_file($file)) {
        $count = (int)trim((string)@file_get_contents($file));
    }
    $count++;
    @file_put_contents($file, (string)$count);

    if ($count > $maxPerMinute) {
        etseo_engine_respond(['ok' => false, 'error' => 'Rate limit'], 429);
    }
}
