<?php
// etseo_engine_ai_client.php
declare(strict_types=1);

require_once __DIR__ . '/etseo_engine_config.php';
require_once __DIR__ . '/etseo_engine_schema.php';
require_once __DIR__ . '/etseo_engine_prompts.php';

function etseo_engine_openai_safe_generate(array $in): array {
    if (!defined('ETSEO_OPENAI_API_KEY') || ETSEO_OPENAI_API_KEY === '') {
        return ['ok' => false, 'error' => 'Missing OpenAI key'];
    }

    $schema = etseo_engine_output_schema();

$model = defined('ETSEO_OPENAI_MODEL') ? ETSEO_OPENAI_MODEL : 'gpt-4o-mini';
$timeout = defined('ETSEO_OPENAI_TIMEOUT') ? (int)ETSEO_OPENAI_TIMEOUT : 30;


    $payload = [
        "model" => ETSEO_OPENAI_MODEL,
        // Responses API: text input
        "input" => [
            [
                "role" => "system",
                "content" => [
                    ["type" => "text", "text" => etseo_engine_safe_system_rules()]
                ]
            ],
            [
                "role" => "user",
                "content" => [
                    ["type" => "text", "text" => etseo_engine_safe_user_prompt($in)]
                ]
            ],
        ],
        // Structured Outputs (JSON schema)
        "text" => [
            "format" => [
                "type" => "json_schema",
                "json_schema" => $schema
            ]
        ],
    ];

    $ch = curl_init('https://api.openai.com/v1/responses');
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/json',
            'Authorization: Bearer ' . ETSEO_OPENAI_API_KEY,
        ],
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_TIMEOUT => (int)ETSEO_OPENAI_TIMEOUT,
    ]);

    $raw = curl_exec($ch);
    $http = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $err = curl_error($ch);
    curl_close($ch);

    if ($raw === false || $raw === '' || $http < 200 || $http >= 300) {
        return ['ok' => false, 'error' => 'OpenAI request failed', 'http' => $http, 'curl' => $err];
    }

    $resp = json_decode($raw, true);
    if (!is_array($resp)) {
        return ['ok' => false, 'error' => 'OpenAI bad JSON'];
    }

    // Extragem textul final din Responses API
    // În practică, output_text e cel mai simplu; dacă nu există, căutăm în "output".
    $text = $resp['output_text'] ?? null;
    if (!is_string($text) || trim($text) === '') {
        // fallback: caută în output[...].content[...].text
        $text = '';
        if (isset($resp['output']) && is_array($resp['output'])) {
            foreach ($resp['output'] as $o) {
                if (!is_array($o)) continue;
                if (($o['type'] ?? '') !== 'message') continue;
                foreach (($o['content'] ?? []) as $c) {
                    if (($c['type'] ?? '') === 'output_text' && isset($c['text'])) {
                        $text .= (string)$c['text'];
                    }
                }
            }
        }
    }

    $text = is_string($text) ? trim($text) : '';
    if ($text === '') return ['ok' => false, 'error' => 'OpenAI empty output'];

    $out = json_decode($text, true);
    if (!is_array($out)) return ['ok' => false, 'error' => 'Model output not JSON'];

    return $out;
}
